<?php
/*
Plugin Name: MDQ Piwigo Integration
Description: Integra fácilmente galerías de Piwigo en tu sitio WordPress con opciones de personalización flexibles.
Version: 1.1
Author: Diego Lazo
Author URI: https://www.mundodequimeras.com
License: GPLv2 or later
Text Domain: mdq-piwigo-integration
Domain Path: /languages
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Load plugin textdomain
function piwigo_integration_load_textdomain() {
    load_plugin_textdomain('piwigo-integration', false, basename(dirname(__FILE__)) . '/languages');
}
add_action('init', 'piwigo_integration_load_textdomain');

// Register a shortcode to display photos
function piwigo_gallery_shortcode($atts) {
    $atts = shortcode_atts(
        array(
            'category' => 1,
            'num_photos' => 24,
            'num_columns' => 4,
        ),
        $atts,
        'piwigo_gallery'
    );

    $category = intval($atts['category']);
    $num_photos = intval($atts['num_photos']);
    $num_columns = intval($atts['num_columns']);
    $api_url = get_option('piwigo_api_url');

    $photos = piwigo_get_photos($category, $api_url, $num_photos);

    if (is_wp_error($photos)) {
        return esc_html__('Error retrieving photos: ', 'piwigo-integration') . esc_html($photos->get_error_message());
    }

    $column_css = "
        <style>
            .piwigo-gallery.piwigo-specific {
                display: grid !important;
                grid-template-columns: repeat({$num_columns}, 1fr) !important;
                gap: 10px !important;
                margin: 0 auto !important;
                max-width: 1200px !important;
            }
            @media (max-width: 1024px) {
                .piwigo-gallery.piwigo-specific {
                    grid-template-columns: repeat(2, 1fr) !important;
                }
            }
            @media (max-width: 768px) {
                .piwigo-gallery.piwigo-specific {
                    grid-template-columns: repeat(1, 1fr) !important;
                }
            }
        </style>
    ";

    $photos = array_slice($photos, 0, $num_photos);

    $output = $column_css . '<div class="piwigo-gallery piwigo-specific">';
    foreach ($photos as $photo) {
        $output .= '<div class="piwigo-photo">';
        $output .= '<a href="' . esc_url($photo['src']) . '" data-lightbox="piwigo-gallery">';
        $output .= '<img src="' . esc_url($photo['src']) . '" alt="' . esc_attr($photo['title']) . '" title="' . esc_attr($photo['title']) . '" loading="lazy">';
        $output .= '</a>';
        $output .= '</div>';
    }
    $output .= '</div>';

    $gallery_url = $api_url . '/index.php?/category/' . $category;

    $output .= '<div class="piwigo-gallery-button">';
    $output .= '<a href="' . esc_url($gallery_url) . '" class="piwigo-view-gallery" target="_blank">' . esc_html__('Ver galería completa', 'piwigo-integration') . '</a>';
    $output .= '</div>';

    return $output;
}
add_shortcode('piwigo_gallery', 'piwigo_gallery_shortcode');

// Function to get photos from Piwigo
function piwigo_get_photos($category, $api_url, $num_photos) {
    $api_url .= '/ws.php?format=json&method=pwg.categories.getImages&cat_id=' . intval($category);

    $response = wp_safe_remote_get($api_url);

    if (is_wp_error($response)) {
        error_log(__('Piwigo API request failed: ', 'piwigo-integration') . $response->get_error_message());
        return $response;
    }

    $body = wp_remote_retrieve_body($response);
    error_log(__('Piwigo API response body: ', 'piwigo-integration') . $body);
    $data = json_decode($body, true);

    if (isset($data['result']['images'])) {
        $photos = array();
        foreach ($data['result']['images'] as $image) {
            $photos[] = array(
                'src' => esc_url_raw($image['element_url']),
                'title' => sanitize_text_field($image['name']),
            );
        }
        return array_slice($photos, 0, $num_photos);
    } else {
        error_log(__('No photos found or invalid API response: ', 'piwigo-integration') . print_r($data, true));
        return new WP_Error('no_photos', __('No photos found.', 'piwigo-integration'));
    }
}

// Enqueue plugin styles and scripts
function piwigo_integration_enqueue_assets() {
    wp_enqueue_style('piwigo-integration-styles', plugin_dir_url(__FILE__) . 'css/style.css');
    wp_enqueue_style('lightbox', plugin_dir_url(__FILE__) . 'css/lightbox.css');
    wp_enqueue_script('lightbox', plugin_dir_url(__FILE__) . 'js/lightbox.js', array('jquery'), null, true);
}
add_action('wp_enqueue_scripts', 'piwigo_integration_enqueue_assets', 20);

// Add Open Graph and Twitter Card meta tags for SEO
function add_opengraph_twitter_meta_tags() {
    if (is_single() && has_shortcode(get_post()->post_content, 'piwigo_gallery')) {
        echo '<meta property="og:title" content="' . get_the_title() . '"/>';
        echo '<meta property="og:type" content="article"/>';
        echo '<meta property="og:url" content="' . get_permalink() . '"/>';
        echo '<meta property="og:image" content="URL_DE_LA_IMAGEN"/>';
        echo '<meta property="og:description" content="' . get_the_excerpt() . '"/>';
        echo '<meta name="twitter:card" content="summary_large_image"/>';
        echo '<meta name="twitter:title" content="' . get_the_title() . '"/>';
        echo '<meta name="twitter:description" content="' . get_the_excerpt() . '"/>';
        echo '<meta name="twitter:image" content="URL_DE_LA_IMAGEN"/>';
    }
}
add_action('wp_head', 'add_opengraph_twitter_meta_tags');

// Add settings page for the plugin
function piwigo_integration_add_admin_menu() {
    add_options_page(
        __('Piwigo Integration Settings', 'piwigo-integration'),
        __('Piwigo Integration', 'piwigo-integration'),
        'manage_options',
        'piwigo-integration',
        'piwigo_integration_options_page'
    );
}
add_action('admin_menu', 'piwigo_integration_add_admin_menu');

// Register settings
function piwigo_integration_settings_init() {
    register_setting('piwigoIntegration', 'piwigo_api_url');
    register_setting('piwigoIntegration', 'piwigo_num_photos');
    register_setting('piwigoIntegration', 'piwigo_num_columns');
    register_setting('piwigoIntegration', 'piwigo_category_id');

    add_settings_section(
        'piwigo_integration_section',
        __('Piwigo Integration Settings', 'piwigo-integration'),
        'piwigo_integration_settings_section_callback',
        'piwigoIntegration'
    );

    add_settings_field(
        'piwigo_api_url',
        __('Piwigo API URL', 'piwigo-integration'),
        'piwigo_api_url_render',
        'piwigoIntegration',
        'piwigo_integration_section',
        array(
            'label_for' => 'piwigo_api_url',
            'description' => __('Ingrese la URL base de la API de su sitio Piwigo. Ejemplo: https://tusitio.com/galeria', 'piwigo-integration')
        )
    );

    add_settings_field(
        'piwigo_num_photos',
        __('Number of Photos', 'piwigo-integration'),
        'piwigo_num_photos_render',
        'piwigoIntegration',
        'piwigo_integration_section',
        array(
            'label_for' => 'piwigo_num_photos',
            'description' => __('Número máximo de fotos para mostrar en la galería.', 'piwigo-integration')
        )
    );

    add_settings_field(
        'piwigo_num_columns',
        __('Number of Columns', 'piwigo-integration'),
        'piwigo_num_columns_render',
        'piwigoIntegration',
        'piwigo_integration_section',
        array(
            'label_for' => 'piwigo_num_columns',
            'description' => __('Número de columnas para la visualización de la galería.', 'piwigo-integration')
        )
    );

    add_settings_field(
        'piwigo_category_id',
        __('Category ID', 'piwigo-integration'),
        'piwigo_category_id_render',
        'piwigoIntegration',
        'piwigo_integration_section',
        array(
            'label_for' => 'piwigo_category_id',
            'description' => __('ID de la categoría de Piwigo para mostrar las fotos.', 'piwigo-integration')
        )
    );
}
add_action('admin_init', 'piwigo_integration_settings_init');

function piwigo_api_url_render($args) {
    $piwigo_api_url = get_option('piwigo_api_url');
    ?>
    <input type='text' name='piwigo_api_url' value='<?php echo esc_url($piwigo_api_url); ?>' size='50'>
    <p class="description"><?php echo esc_html($args['description']); ?></p>
    <?php
}

function piwigo_num_photos_render($args) {
    $piwigo_num_photos = get_option('piwigo_num_photos', 24);
    ?>
    <input type='number' name='piwigo_num_photos' value='<?php echo intval($piwigo_num_photos); ?>' min='1'>
    <p class="description"><?php echo esc_html($args['description']); ?></p>
    <?php
}

function piwigo_num_columns_render($args) {
    $piwigo_num_columns = get_option('piwigo_num_columns', 4);
    ?>
    <input type='number' name='piwigo_num_columns' value='<?php echo intval($piwigo_num_columns); ?>' min='1'>
    <p class="description"><?php echo esc_html($args['description']); ?></p>
    <?php
}

function piwigo_category_id_render($args) {
    $piwigo_category_id = get_option('piwigo_category_id', 1);
    ?>
    <input type='number' name='piwigo_category_id' value='<?php echo intval($piwigo_category_id); ?>' min='1'>
    <p class="description"><?php echo esc_html($args['description']); ?></p>
    <?php
}

function piwigo_integration_settings_section_callback() {
    echo __('Ingrese la URL base para su API de Piwigo, el número de fotos a mostrar, el número de columnas y el ID de la categoría. Ejemplo: https://tusitio.com/galeria', 'piwigo-integration');
}

function piwigo_integration_options_page() {
    ?>
    <form action='options.php' method='post' id='piwigo-integration-form'>
        <h2><?php _e('Piwigo Integration Settings', 'piwigo-integration'); ?></h2>
        <?php
        settings_fields('piwigoIntegration');
        do_settings_sections('piwigoIntegration');
        submit_button();
        ?>
        <form method="post">
            <input type="hidden" name="piwigo_reset_defaults" value="1" />
            <?php submit_button(__('Restablecer a Predeterminados', 'piwigo-integration'), 'secondary'); ?>
        </form>
        <h2><?php _e('Importar/Exportar Configuraciones', 'piwigo-integration'); ?></h2>
        <form method="post" enctype="multipart/form-data">
            <input type="hidden" name="piwigo_import_settings" value="1" />
            <input type="file" name="piwigo_settings_file" />
            <?php submit_button(__('Importar Configuraciones', 'piwigo-integration'), 'secondary'); ?>
        </form>
        <form method="post">
            <input type="hidden" name="piwigo_export_settings" value="1" />
            <?php submit_button(__('Exportar Configuraciones', 'piwigo-integration'), 'secondary'); ?>
        </form>
    </form>
    <h2><?php _e('Vista Previa', 'piwigo-integration'); ?></h2>
    <div id="piwigo-integration-preview">
        <?php echo piwigo_get_preview(); ?>
    </div>
    <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#piwigo-integration-form input').on('change', function() {
                var data = {
                    action: 'piwigo_integration_preview',
                    piwigo_api_url: $('input[name="piwigo_api_url"]').val(),
                    piwigo_num_photos: $('input[name="piwigo_num_photos"]').val(),
                    piwigo_num_columns: $('input[name="piwigo_num_columns"]').val(),
                    piwigo_category_id: $('input[name="piwigo_category_id"]').val()
                };

                $.post(ajaxurl, data, function(response) {
                    $('#piwigo-integration-preview').html(response);
                });
            });
        });
    </script>
    <?php
}

// Función para restablecer configuraciones a los valores predeterminados
function piwigo_integration_reset_defaults() {
    if (isset($_POST['piwigo_reset_defaults']) && $_POST['piwigo_reset_defaults'] == '1') {
        update_option('piwigo_api_url', '');
        update_option('piwigo_num_photos', 24);
        update_option('piwigo_num_columns', 4);
        update_option('piwigo_category_id', 1);
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success is-dismissible"><p>' . __('Configuraciones restablecidas a los valores predeterminados.', 'piwigo-integration') . '</p></div>';
        });
    }
}
add_action('admin_init', 'piwigo_integration_reset_defaults');

// Función para exportar configuraciones
function piwigo_integration_export_settings() {
    if (isset($_POST['piwigo_export_settings']) && $_POST['piwigo_export_settings'] == '1') {
        $settings = array(
            'piwigo_api_url' => get_option('piwigo_api_url'),
            'piwigo_num_photos' => get_option('piwigo_num_photos'),
            'piwigo_num_columns' => get_option('piwigo_num_columns'),
            'piwigo_category_id' => get_option('piwigo_category_id')
        );

        $json_settings = json_encode($settings);
        header('Content-disposition: attachment; filename=piwigo-settings.json');
        header('Content-type: application/json');
        echo $json_settings;
        exit;
    }
}
add_action('admin_init', 'piwigo_integration_export_settings');

// Función para importar configuraciones
function piwigo_integration_import_settings() {
    if (isset($_POST['piwigo_import_settings']) && $_POST['piwigo_import_settings'] == '1') {
        if (isset($_FILES['piwigo_settings_file']) && $_FILES['piwigo_settings_file']['error'] == 0) {
            $json_settings = file_get_contents($_FILES['piwigo_settings_file']['tmp_name']);
            $settings = json_decode($json_settings, true);

            if (json_last_error() === JSON_ERROR_NONE) {
                if (isset($settings['piwigo_api_url']) && isset($settings['piwigo_num_photos']) && isset($settings['piwigo_num_columns']) && isset($settings['piwigo_category_id'])) {
                    update_option('piwigo_api_url', sanitize_text_field($settings['piwigo_api_url']));
                    update_option('piwigo_num_photos', intval($settings['piwigo_num_photos']));
                    update_option('piwigo_num_columns', intval($settings['piwigo_num_columns']));
                    update_option('piwigo_category_id', intval($settings['piwigo_category_id']));
                    add_action('admin_notices', function() {
                        echo '<div class="notice notice-success is-dismissible"><p>' . __('Configuraciones importadas correctamente.', 'piwigo-integration') . '</p></div>';
                    });
                } else {
                    add_action('admin_notices', function() {
                        echo '<div class="notice notice-error is-dismissible"><p>' . __('Error al importar configuraciones. Faltan campos requeridos.', 'piwigo-integration') . '</p></div>';
                    });
                }
            } else {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-error is-dismissible"><p>' . __('Error al importar configuraciones. Archivo JSON no válido.', 'piwigo-integration') . '</p></div>';
                });
            }
        } else {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>' . __('Error al subir el archivo de configuraciones.', 'piwigo-integration') . '</p></div>';
            });
        }
    }
}
add_action('admin_init', 'piwigo_integration_import_settings');

function piwigo_get_preview() {
    $api_url = get_option('piwigo_api_url');
    $num_photos = get_option('piwigo_num_photos', 24);
    $num_columns = get_option('piwigo_num_columns', 4);
    $category_id = get_option('piwigo_category_id', 1);

    if ($api_url) {
        $photos = piwigo_get_photos($category_id, $api_url, $num_photos);

        if (!is_wp_error($photos)) {
            $column_css = "
                <style>
                    .piwigo-gallery.piwigo-preview {
                        display: grid !important;
                        grid-template-columns: repeat({$num_columns}, 1fr) !important;
                        gap: 10px !important;
                        margin: 0 auto !important;
                        max-width: 1200px !important;
                    }
                    .piwigo-gallery.piwigo-preview .piwigo-photo {
                        text-align: center !important;
                    }
                    .piwigo-gallery.piwigo-preview .piwigo-photo img {
                        width: 100% !important;
                        height: auto !important;
                        display: block !important;
                    }
                    @media (max-width: 1024px) {
                        .piwigo-gallery.piwigo-preview {
                            grid-template-columns: repeat(2, 1fr) !important;
                        }
                    }
                    @media (max-width: 768px) {
                        .piwigo-gallery.piwigo-preview {
                            grid-template-columns: repeat(1, 1fr) !important;
                        }
                    }
                </style>
            ";

            $photos = array_slice($photos, 0, intval($num_photos));

            $output = $column_css . '<div class="piwigo-gallery piwigo-preview">';
            foreach ($photos as $photo) {
                $output .= '<div class="piwigo-photo">';
                $output .= '<a href="' . esc_url($photo['src']) . '" data-lightbox="piwigo-preview">';
                $output .= '<img src="' . esc_url($photo['src']) . '" alt="' . esc_attr($photo['title']) . '" title="' . esc_attr($photo['title']) . '" loading="lazy">';
                $output .= '</a>';
                $output .= '</div>';
            }
            $output .= '</div>';

            return $output;
        }
    }

    return '<p>' . esc_html__('Configura la URL de la API para ver la vista previa.', 'piwigo-integration') . '</p>';
}

add_action('wp_ajax_piwigo_integration_preview', 'piwigo_integration_preview_ajax');

function piwigo_integration_preview_ajax() {
    $api_url = isset($_POST['piwigo_api_url']) ? esc_url_raw($_POST['piwigo_api_url']) : '';
    $num_photos = isset($_POST['piwigo_num_photos']) ? intval($_POST['piwigo_num_photos']) : 24;
    $num_columns = isset($_POST['piwigo_num_columns']) ? intval($_POST['piwigo_num_columns']) : 4;
    $category_id = isset($_POST['piwigo_category_id']) ? intval($_POST['piwigo_category_id']) : 1;

    echo piwigo_get_preview_html($api_url, $num_photos, $num_columns, $category_id);
    wp_die();
}

function piwigo_get_preview_html($api_url, $num_photos, $num_columns, $category_id) {
    if ($api_url) {
        $photos = piwigo_get_photos($category_id, $api_url, $num_photos);

        if (!is_wp_error($photos)) {
            $column_css = "
                <style>
                    .piwigo-gallery.piwigo-preview {
                        display: grid !important;
                        grid-template-columns: repeat({$num_columns}, 1fr) !important;
                        gap: 10px !important;
                        margin: 0 auto !important;
                        max-width: 1200px !important;
                    }
                    .piwigo-gallery.piwigo-preview .piwigo-photo {
                        text-align: center !important;
                    }
                    .piwigo-gallery.piwigo-preview .piwigo-photo img {
                        width: 100% !important;
                        height: auto !important;
                        display: block !important;
                    }
                    @media (max-width: 1024px) {
                        .piwigo-gallery.piwigo-preview {
                            grid-template-columns: repeat(2, 1fr) !important;
                        }
                    }
                    @media (max-width: 768px) {
                        .piwigo-gallery.piwigo-preview {
                            grid-template-columns: repeat(1, 1fr) !important;
                        }
                    }
                </style>
            ";

            $photos = array_slice($photos, 0, intval($num_photos));

            $output = $column_css . '<div class="piwigo-gallery piwigo-preview">';
            foreach ($photos as $photo) {
                $output .= '<div class="piwigo-photo">';
                $output .= '<a href="' . esc_url($photo['src']) . '" data-lightbox="piwigo-preview">';
                $output .= '<img src="' . esc_url($photo['src']) . '" alt="' . esc_attr($photo['title']) . '" title="' . esc_attr($photo['title']) . '" loading="lazy">';
                $output .= '</a>';
                $output .= '</div>';
            }
            $output .= '</div>';

            return $output;
        }
    }

    return '<p>' . esc_html__('Configura la URL de la API para ver la vista previa.', 'piwigo-integration') . '</p>';
}

// Enqueue block editor assets for Gutenberg
function piwigo_integration_enqueue_block_editor_assets() {
    wp_enqueue_script(
        'piwigo-integration-block',
        plugins_url( 'block.js', __FILE__ ),
        array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components' ),
        filemtime( plugin_dir_path( __FILE__ ) . 'block.js' )
    );

    wp_enqueue_style(
        'piwigo-integration-block-editor',
        plugins_url( 'block-editor.css', __FILE__ ),
        array( 'wp-edit-blocks' ),
        filemtime( plugin_dir_path( __FILE__ ) . 'block-editor.css' )
    );
}
add_action( 'enqueue_block_editor_assets', 'piwigo_integration_enqueue_block_editor_assets' );

// Register block editor assets
function piwigo_integration_register_block() {
    wp_register_script(
        'piwigo-integration-block',
        plugins_url( 'block.js', __FILE__ ),
        array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components' ),
        filemtime( plugin_dir_path( __FILE__ ) . 'block.js' )
    );
}
add_action( 'init', 'piwigo_integration_register_block' );
?>
