<?php
/*
Plugin Name: MDQ MAPS
Plugin URI: https://www.mundodequimeras.com
Description: Un plugin para agregar y mostrar puntos personalizados en un mapa de Google Maps.
Version: 1.0
Author: Diego Lazo
Author URI: https://www.mundodequimeras.com
License: GPL2
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Registrar los scripts necesarios para el frontend y backend
function mdq_maps_register_scripts()
{
    wp_enqueue_script('google-maps-api', 'https://maps.googleapis.com/maps/api/js?key=' . esc_attr(get_option('google_maps_api_key')) . '&libraries=places&callback=initMap', array(), null, true);
}
add_action('wp_enqueue_scripts', 'mdq_maps_register_scripts');

// Cargar los scripts de la biblioteca de medios y jQuery en el backend
function mdq_maps_enqueue_media_uploader($hook)
{
    if ('toplevel_page_mdq-maps-points' === $hook) {
        wp_enqueue_media();
        wp_enqueue_script('jquery');
    }
}
add_action('admin_enqueue_scripts', 'mdq_maps_enqueue_media_uploader');

// Registrar la taxonomía de Categorías para Puntos del Mapa
function mdq_maps_register_taxonomy()
{
    $labels = array(
        'name' => _x('Categorías de Puntos', 'taxonomy general name'),
        'singular_name' => _x('Categoría de Punto', 'taxonomy singular name'),
        'search_items' => __('Buscar Categorías de Puntos'),
        'all_items' => __('Todas las Categorías de Puntos'),
        'parent_item' => __('Categoría Padre'),
        'parent_item_colon' => __('Categoría Padre:'),
        'edit_item' => __('Editar Categoría de Punto'),
        'update_item' => __('Actualizar Categoría de Punto'),
        'add_new_item' => __('Agregar Nueva Categoría de Punto'),
        'new_item_name' => __('Nuevo Nombre de Categoría de Punto'),
        'menu_name' => __('Categorías de Puntos'),
    );

    $args = array(
        'hierarchical' => true,
        'labels' => $labels,
        'show_ui' => true,
        'show_admin_column' => true,
        'query_var' => true,
        'rewrite' => array('slug' => 'mdq_maps_category'),
    );

    register_taxonomy('mdq_maps_category', array('mdq_maps_points'), $args);
}
add_action('init', 'mdq_maps_register_taxonomy');

// Crear el menú en el panel de administración
function mdq_maps_menu()
{
    add_menu_page(
        'MDQ MAPS',               // Título de la página
        'MDQ MAPS',               // Título del menú
        'manage_options',         // Capacidad
        'mdq-maps-points',        // Slug (ahora apunta a la página de puntos)
        'mdq_maps_points_page',   // Función para mostrar el contenido (de la pantalla de puntos)
        'dashicons-location-alt'  // Icono del menú
    );
    add_submenu_page(
        'mdq-maps-points',       // Hace que este submenú sea el principal
        'Configuración de API',
        'Configuración de API',
        'manage_options',
        'mdq-maps-api-settings',
        'mdq_maps_api_settings_page'
    );
    add_submenu_page(
        'mdq-maps-points',        // El slug del menú principal también es el parent_slug de los submenús
        'Cómo Insertar el Mapa',
        'Cómo Insertar el Mapa',
        'manage_options',
        'mdq-maps-how-to',
        'mdq_maps_how_to_page'
    );
    // Submenú para gestionar las categorías
    add_submenu_page(
        'mdq-maps-points',
        'Categorías de Puntos',
        'Categorías de Puntos',
        'manage_options',
        'edit-tags.php?taxonomy=mdq_maps_category&post_type=mdq_maps_points'
    );
}
add_action('admin_menu', 'mdq_maps_menu');

// Página de Configuración de la API
function mdq_maps_api_settings_page()
{
    if (!current_user_can('manage_options')) {
        wp_die(__('Lo siento, no tienes permisos para acceder a esta página.'));
    }
    ?>
    <div class="wrap">
        <h1>Configuración de la API de Google Maps</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('mdq_maps_settings_group');
            do_settings_sections('mdq_maps_settings_group');
            ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">Clave API de Google Maps</th>
                    <td>
                        <input type="text" name="google_maps_api_key" value="<?php echo esc_attr(get_option('google_maps_api_key')); ?>" style="width: 100%;" />
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

// Registrar la opción y la configuración de la API
function mdq_maps_register_settings()
{
    register_setting('mdq_maps_settings_group', 'google_maps_api_key');
}
add_action('admin_init', 'mdq_maps_register_settings');

// Página de "Cómo Insertar el Mapa"
function mdq_maps_how_to_page()
{
    ?>
    <div class="wrap">
        <h1>Cómo Insertar el Mapa</h1>
        <p>Para insertar el mapa en una página o artículo, utiliza el siguiente shortcode:</p>
        <code>[mdq_maps]</code>
        <p>Para mostrar puntos de una categoría específica, usa:</p>
        <code>[mdq_maps category="nombre-de-la-categoria"]</code>
        <p>Reemplaza <em>nombre-de-la-categoria</em> con el slug de la categoría que deseas mostrar.</p>
        <p>Para permitir a los usuarios filtrar por categoría, usa:</p>
        <code>[mdq_maps filter="true"]</code>
    </div>
    <?php
}

// Crear la tabla en la base de datos al activar el plugin
function mdq_maps_create_table()
{
    global $wpdb;
    $table_name = $wpdb->prefix . 'mdq_maps_points';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        title text NOT NULL,
        image text,
        link text NOT NULL,
        lat float NOT NULL,
        lng float NOT NULL,
        category VARCHAR(255) NOT NULL,
        PRIMARY KEY  (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'mdq_maps_create_table');

// Función para actualizar la tabla y agregar la columna faltante
function mdq_maps_update_table()
{
    global $wpdb;
    $table_name = $wpdb->prefix . 'mdq_maps_points';

    // Añadir la columna 'category' si no existe
    $wpdb->query("ALTER TABLE $table_name ADD COLUMN category VARCHAR(255) NOT NULL AFTER lng");
}
add_action('admin_init', 'mdq_maps_update_table');

// Función para borrar todos los puntos
function mdq_maps_delete_all_points()
{
    if (!current_user_can('manage_options')) {
        wp_die(__('Lo siento, no tienes permisos para acceder a esta página.'));
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'mdq_maps_points';

    if (isset($_POST['mdq_delete_all_points']) && check_admin_referer('mdq_delete_all_points_action')) {
        $wpdb->query("TRUNCATE TABLE $table_name");
        echo '<div class="updated"><p>Todos los puntos han sido eliminados.</p></div>';
    }
}

// Página de Gestión de Puntos en el Mapa con Mapa Interactivo
function mdq_maps_points_page()
{
    if (!current_user_can('manage_options')) {
        wp_die(__('Lo siento, no tienes permisos para acceder a esta página.'));
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'mdq_maps_points';

    // Si se elimina un punto
    if (isset($_GET['action']) && $_GET['action'] == 'delete' && isset($_GET['id'])) {
        $id = intval($_GET['id']);
        check_admin_referer('delete_point_' . $id);

        $wpdb->delete($table_name, array('id' => $id));
        echo '<div class="updated"><p>Punto eliminado correctamente.</p></div>';
    }

    // Eliminar todos los puntos
    mdq_maps_delete_all_points();

    // Si se edita o agrega un nuevo punto
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && !isset($_POST['mdq_delete_all_points'])) {
        $id = isset($_POST['point_id']) ? intval($_POST['point_id']) : null;
        check_admin_referer('save_point_' . ($id ? $id : 'new'));

        $title = sanitize_text_field($_POST['point_title']);
        $image = !empty($_POST['point_image']) ? esc_url_raw($_POST['point_image']) : null;
        $link = esc_url_raw($_POST['point_link']);
        $lat = floatval($_POST['point_lat']);
        $lng = floatval($_POST['point_lng']);
        $category = sanitize_text_field($_POST['point_category']);

        // Crear el archivo de log en wp-content para depuración
        $log_file = ABSPATH . 'wp-content/debug_log.txt';
        file_put_contents($log_file, "Intentando guardar el punto: " . print_r($_POST, true) . "\n", FILE_APPEND);

        // Validar que los campos requeridos estén completos, incluida la categoría
        if (empty($title) || empty($lat) || empty($lng) || empty($category)) {
            echo '<div class="error"><p>Por favor completa todos los campos requeridos: Título, Latitud, Longitud, y Categoría.</p></div>';
            file_put_contents($log_file, "Fallo en la validación: campos vacíos.\n", FILE_APPEND);
        } else {
            $data = array(
                'title' => $title,
                'link' => $link,
                'lat' => $lat,
                'lng' => $lng,
                'category' => $category
            );

            if ($image) {
                $data['image'] = $image;
            }

            if ($id) {
                $result = $wpdb->update(
                    $table_name,
                    $data,
                    array('id' => $id)
                );
                file_put_contents($log_file, "Consulta de actualización: " . $wpdb->last_query . "\n", FILE_APPEND);
                file_put_contents($log_file, "Resultado de la actualización: " . print_r($result, true) . "\n", FILE_APPEND);
                echo '<div class="updated"><p>Punto actualizado correctamente.</p></div>';
                echo '<script type="text/javascript">document.location.href="' . admin_url('admin.php?page=mdq-maps-points') . '";</script>';
                exit;
            } else {
                $result = $wpdb->insert($table_name, $data);
                file_put_contents($log_file, "Consulta de inserción: " . $wpdb->last_query . "\n", FILE_APPEND);
                file_put_contents($log_file, "Resultado de la inserción: " . print_r($result, true) . "\n", FILE_APPEND);
                file_put_contents($log_file, "Error de MySQL: " . $wpdb->last_error . "\n", FILE_APPEND);
                if ($result) {
                    echo '<div class="updated"><p>Punto agregado correctamente.</p></div>';
                } else {
                    echo '<div class="error"><p>Hubo un error al intentar guardar el punto. Revisa los logs para más detalles.</p></div>';
                }
            }
        }
    }

    // Obtener los puntos actuales y aplicar la búsqueda y filtro por categoría
    $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
    $category_filter = isset($_GET['category_filter']) ? sanitize_text_field($_GET['category_filter']) : '';

    $query = "SELECT * FROM $table_name";
    $where_clauses = array();

    if ($search) {
        $where_clauses[] = $wpdb->prepare("title LIKE %s", '%' . $wpdb->esc_like($search) . '%');
    }

    if ($category_filter && $category_filter !== 'all') {
        $where_clauses[] = $wpdb->prepare("category = %s", $category_filter);
    }

    if (!empty($where_clauses)) {
        $query .= " WHERE " . implode(" AND ", $where_clauses);
    }

    $points = $wpdb->get_results($query);

    // Formulario para agregar o editar puntos
    $editing = isset($_GET['action']) && $_GET['action'] == 'edit' && isset($_GET['id']);
    if ($editing) {
        $edit_id = intval($_GET['id']);
        check_admin_referer('edit_point_' . $edit_id);
        $point = $wpdb->get_row("SELECT * FROM $table_name WHERE id = $edit_id");
    }
    ?>
    <div class="wrap">
        <h1><?php echo $editing ? 'Editar Punto' : 'Agregar Nuevo Punto'; ?></h1>

        <form method="post" action="">
            <?php if ($editing): ?>
                <input type="hidden" name="point_id" value="<?php echo esc_attr($point->id); ?>" />
                <?php wp_nonce_field('save_point_' . $point->id); ?>
            <?php else: ?>
                <?php wp_nonce_field('save_point_new'); ?>
            <?php endif; ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">Título</th>
                    <td><input type="text" name="point_title" value="<?php echo $editing ? esc_attr($point->title) : ''; ?>" required /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Imagen (URL)</th>
                    <td>
                        <input type="text" name="point_image" id="point_image" value="<?php echo $editing ? esc_attr($point->image) : ''; ?>" style="width: 80%;" />
                        <input type="button" class="button button-secondary" value="Seleccionar Imagen" id="upload_image_button" />
                        <img id="point_image_preview" src="<?php echo $editing ? esc_attr($point->image) : ''; ?>" style="max-width: 200px; display: <?php echo $editing && $point->image ? 'block' : 'none'; ?>; margin-top: 10px;" />
                        <p class="description">Ingresa la URL de la imagen o selecciona una desde la biblioteca de medios.</p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">Vínculo</th>
                    <td><input type="url" name="point_link" value="<?php echo $editing ? esc_attr($point->link) : ''; ?>" required /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Buscar Dirección</th>
                    <td><input type="text" id="map-search" placeholder="Ingresa una dirección o lugar..." style="width: 100%;" /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Latitud</th>
                    <td><input type="text" name="point_lat" id="point_lat" value="<?php echo $editing ? esc_attr($point->lat) : ''; ?>" required /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Longitud</th>
                    <td><input type="text" name="point_lng" id="point_lng" value="<?php echo $editing ? esc_attr($point->lng) : ''; ?>" required /></td>
                </tr>
                <tr valign="top">
                    <th scope="row">Categoría</th>
                    <td>
                        <?php
                        wp_dropdown_categories(array(
                            'taxonomy' => 'mdq_maps_category',
                            'name' => 'point_category',
                            'selected' => $editing ? esc_attr($point->category) : '',
                            'show_option_none' => __('Seleccionar Categoría'),
                            'hide_empty' => false, // Mostrar categorías vacías
                            'required' => true // Campo obligatorio
                        ));
                        ?>
                    </td>
                </tr>
            </table>
            <div id="mdq-map-canvas" style="width: 100%; height: 400px;"></div>
            <?php submit_button($editing ? 'Actualizar Punto' : 'Guardar Punto'); ?>
            <?php if ($editing): ?>
                <a href="<?php echo admin_url('admin.php?page=mdq-maps-points'); ?>" class="button button-secondary">Cancelar</a>
            <?php endif; ?>
        </form>

        <h2>Puntos Actuales en el Mapa</h2>
        <form method="get" action="">
            <input type="hidden" name="page" value="mdq-maps-points" />
            <input type="text" name="s" value="<?php echo esc_attr($search); ?>" placeholder="Buscar puntos..." />
            <?php
            wp_dropdown_categories(array(
                'taxonomy' => 'mdq_maps_category',
                'name' => 'category_filter',
                'show_option_all' => __('Todas las Categorías'),
                'selected' => $category_filter,
                'hide_empty' => false, // Mostrar categorías vacías en el filtro
            ));
            ?>
            <input type="submit" value="Filtrar" class="button" />
            <a href="<?php echo admin_url('admin.php?page=mdq-maps-points'); ?>" class="button">Limpiar Filtros</a>
        </form>

        <form method="post" action="">
            <?php wp_nonce_field('mdq_delete_all_points_action'); ?>
            <input type="hidden" name="mdq_delete_all_points" value="1" />
            <input type="submit" value="Borrar Todos los Puntos" class="button button-danger" onclick="return confirm('¿Estás seguro de que deseas eliminar todos los puntos? Esta acción no se puede deshacer.');" />
        </form>

        <table class="widefat">
            <thead>
                <tr>
                    <th>Título</th>
                    <th>Imagen</th>
                    <th>Vínculo</th>
                    <th>Latitud</th>
                    <th>Longitud</th>
                    <th>Categoría</th>
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($points): ?>
                    <?php foreach ($points as $point) : ?>
                        <tr>
                            <td><?php echo esc_html($point->title); ?></td>
                            <td>
                                <?php if ($point->image): ?>
                                    <img src="<?php echo esc_url($point->image); ?>" style="max-width: 100px;" />
                                <?php else: ?>
                                    <img src="<?php echo plugins_url('default-marker.png', __FILE__); ?>" style="max-width: 50px;" />
                                <?php endif; ?>
                            </td>
                            <td><a href="<?php echo esc_url($point->link); ?>" target="_blank"><?php echo esc_url($point->link); ?></a></td>
                            <td><?php echo esc_html($point->lat); ?></td>
                            <td><?php echo esc_html($point->lng); ?></td>
                            <td>
                                <?php 
                                $category = get_term($point->category, 'mdq_maps_category');
                                echo esc_html($category->name); 
                                ?>
                            </td>
                            <td>
                                <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=mdq-maps-points&action=edit&id=' . $point->id), 'edit_point_' . $point->id); ?>">Editar</a> | 
                                <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=mdq-maps-points&action=delete&id=' . $point->id), 'delete_point_' . $point->id); ?>" onclick="return confirm('¿Estás seguro de que deseas eliminar este punto?');">Eliminar</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr><td colspan="7">No se encontraron puntos en el mapa.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#upload_image_button').click(function(e) {
                e.preventDefault();

                var image = wp.media({
                    title: 'Seleccionar Imagen',
                    multiple: false // Permitir selección de una sola imagen
                }).open()
                .on('select', function() {
                    var uploaded_image = image.state().get('selection').first();
                    var image_url = uploaded_image.toJSON().url;
                    $('#point_image').val(image_url);
                    $('#point_image_preview').attr('src', image_url).show();
                });
            });

            // Cargar el mapa
            initAdminMap();
        });

        function initAdminMap() {
            var map = new google.maps.Map(document.getElementById('mdq-map-canvas'), {
                zoom: 8,
                center: {lat: -32.5228, lng: -55.7658} // Coordenadas centradas en Uruguay
            });

            var marker = new google.maps.Marker({
                position: {lat: -32.5228, lng: -55.7658},
                map: map,
                draggable: true
            });

            // Autocompletar dirección
            var input = document.getElementById('map-search');
            var autocomplete = new google.maps.places.Autocomplete(input);
            autocomplete.bindTo('bounds', map);

            autocomplete.addListener('place_changed', function() {
                var place = autocomplete.getPlace();
                if (!place.geometry) {
                    return;
                }

                // Si el lugar tiene una geometría, centro el mapa en ella y ajusto el marcador
                if (place.geometry.viewport) {
                    map.fitBounds(place.geometry.viewport);
                } else {
                    map.setCenter(place.geometry.location);
                    map.setZoom(17);  // Zoom para mostrar el lugar en detalle
                }

                marker.setPosition(place.geometry.location);
                document.getElementById('point_lat').value = place.geometry.location.lat();
                document.getElementById('point_lng').value = place.geometry.location.lng();
            });

            google.maps.event.addListener(map, 'click', function(event) {
                marker.setPosition(event.latLng);
                document.getElementById('point_lat').value = event.latLng.lat();
                document.getElementById('point_lng').value = event.latLng.lng();
            });

            google.maps.event.addListener(marker, 'dragend', function(event) {
                document.getElementById('point_lat').value = event.latLng.lat();
                document.getElementById('point_lng').value = event.latLng.lng();
            });

            // Si se están editando puntos, centra el mapa en las coordenadas del punto
            <?php if ($editing): ?>
                var editLatLng = {lat: <?php echo $point->lat; ?>, lng: <?php echo $point->lng; ?>};
                map.setCenter(editLatLng);
                marker.setPosition(editLatLng);
            <?php endif; ?>
        }
    </script>
    <script src="https://maps.googleapis.com/maps/api/js?key=<?php echo esc_attr(get_option('google_maps_api_key')); ?>&libraries=places"></script>
    <?php
}

// Shortcode para mostrar el mapa en el frontend con opción de filtro
function mdq_maps_display_map_shortcode($atts)
{
    global $wpdb;
    $table_name = $wpdb->prefix . 'mdq_maps_points';

    // Procesar los atributos del shortcode
    $atts = shortcode_atts(array(
        'category' => '',
        'filter' => 'false', // Nuevo atributo para mostrar filtro de categorías
    ), $atts, 'mdq_maps');

    // Crear un contenedor para el mapa
    $output = '';

    // Si se habilita el filtro de categorías
    if ($atts['filter'] === 'true') {
        $categories = get_terms(array(
            'taxonomy' => 'mdq_maps_category',
            'hide_empty' => false, // Cambiar esto para que muestre todas las categorías, incluso las vacías
        ));

        if (!empty($categories)) {
            $output .= '<div id="mdq-category-filter" style="margin-bottom: 10px;"><strong>Filtrar por Categoría:</strong> ';
            $output .= '<select id="mdq-category-select"><option value="all">Todas</option>';
            foreach ($categories as $category) {
                $output .= '<option value="' . esc_attr($category->term_id) . '">' . esc_html($category->name) . '</option>';
            }
            $output .= '</select></div>';
        }
    }

    $output .= '<div id="mdq-google-map" style="width: 100%; height: 500px;"></div>';

    // Preparar los datos de los puntos en formato JSON para pasarlos a JavaScript
    $query = "SELECT * FROM $table_name";
    $where_clauses = array();

    if (!empty($atts['category'])) {
        $where_clauses[] = $wpdb->prepare("category = %s", sanitize_text_field($atts['category']));
    }

    if (!empty($where_clauses)) {
        $query .= " WHERE " . implode(" AND ", $where_clauses);
    }

    $points = $wpdb->get_results($query);
    $points_data = json_encode($points);

    // Incluir un script para inicializar el mapa y los puntos
    $output .= "
        <script type='text/javascript'>
        function initMap() {
            var map = new google.maps.Map(document.getElementById('mdq-google-map'), {
                zoom: 8,
                center: {lat: -32.5228, lng: -55.7658} // Coordenadas centradas en Uruguay
            });

            var infowindow = new google.maps.InfoWindow();
            var defaultIcon = '" . plugins_url('default-marker.png', __FILE__) . "';

            var points = $points_data;
            var markers = [];

            points.forEach(function(point) {
                var marker = new google.maps.Marker({
                    position: {lat: parseFloat(point.lat), lng: parseFloat(point.lng)},
                    map: map,
                    title: point.title,
                    icon: {
                        url: point.image ? point.image : defaultIcon,
                        scaledSize: new google.maps.Size(50, 50),
                        labelOrigin: new google.maps.Point(25, 25),
                        anchor: new google.maps.Point(25, 25)
                    }
                });

                marker.set('category', point.category); // Establecer la categoría del marcador

                marker.addListener('click', function() {
                    infowindow.setContent('<div><img src=\"' + (point.image ? point.image : defaultIcon) + '\" alt=\"' + point.title + '\" style=\"width:100px;height:auto;\"><br><strong>' + point.title + '</strong><br><a href=\"' + point.link + '\" target=\"_blank\">Escuchar Radio</a></div>');
                    infowindow.open(map, marker);
                });

                markers.push(marker);
            });

            // Función para filtrar los puntos por categoría
            function filterMarkers(category) {
                markers.forEach(function(marker) {
                    if (category === 'all' || marker.get('category') == category) {
                        marker.setVisible(true);
                    } else {
                        marker.setVisible(false);
                    }
                });
            }

            // Si el filtro está activado, agregar evento de cambio para el select
            if (document.getElementById('mdq-category-select')) {
                document.getElementById('mdq-category-select').addEventListener('change', function() {
                    var selectedCategory = this.value;
                    filterMarkers(selectedCategory);
                });
            }
        }

        // Cargar el mapa al asegurarnos de que la página esté completamente cargada
        document.addEventListener('DOMContentLoaded', function() {
            initMap();
        });
        </script>
    ";

    return $output;
}
add_shortcode('mdq_maps', 'mdq_maps_display_map_shortcode');
