<?php
/**
 * Plugin Name: MDQ Sitio en Mantenimiento
 * Plugin URI: https://www.mundodequimeras.com
 * Description: Plugin para modo mantenimiento con opciones de personalización y programación.
 * Version: 1.3
 * Author: Diego Lazo
 * Author URI: https://www.mundodequimeras.com
 * License: GPL2
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class MDQ_Sitio_Mantenimiento {

    public function __construct() {
        add_action('admin_menu', [$this, 'crear_menu']);
        add_action('template_redirect', [$this, 'mostrar_mantenimiento']);
        add_action('wp_loaded', [$this, 'verificar_programacion']);
        add_action('admin_enqueue_scripts', [$this, 'cargar_scripts']);
    }

    public function crear_menu() {
        add_menu_page(
            'Modo Mantenimiento',
            'Modo Mantenimiento',
            'manage_options',
            'mdq-mantenimiento',
            [$this, 'panel_admin'],
            'dashicons-hammer',
            50
        );
    }

    public function panel_admin() {
        if (!current_user_can('manage_options')) {
            return;
        }

        // Procesar datos antes de generar cualquier salida
        if (isset($_POST['mdq_guardar'])) {
            update_option('mdq_modo_bloqueo', sanitize_text_field(wp_unslash($_POST['mdq_modo_bloqueo'] ?? 'todo')));
            update_option('mdq_categorias_bloqueadas', array_map('intval', $_POST['mdq_categorias_bloqueadas'] ?? []));
            update_option('mdq_paginas_bloqueadas', array_map('intval', $_POST['mdq_paginas_bloqueadas'] ?? []));
            update_option('mdq_modo_manual', isset($_POST['mdq_modo_manual']));
            update_option('mdq_mensaje_mantenimiento', sanitize_text_field(wp_unslash($_POST['mdq_mensaje_mantenimiento'] ?? '')));
            update_option('mdq_imagen_mantenimiento', esc_url_raw(wp_unslash($_POST['mdq_imagen_mantenimiento'] ?? '')));
            update_option('mdq_inicio_mantenimiento', sanitize_text_field(wp_unslash($_POST['mdq_inicio_mantenimiento'] ?? '')));
            update_option('mdq_fin_mantenimiento', sanitize_text_field(wp_unslash($_POST['mdq_fin_mantenimiento'] ?? '')));

            // Mensaje de confirmación
            add_settings_error(
                'mdq_mantenimiento_messages',
                'mdq_mantenimiento_message',
                esc_html__('Los cambios se guardaron correctamente.', 'mdq-sitio-en-mantenimiento'),
                'updated'
            );
        }

        // Incluir HTML de administración
        include plugin_dir_path(__FILE__) . 'admin-panel.php';
    }

    public function mostrar_mantenimiento() {
    // No bloquear si es un administrador
    if (current_user_can('manage_options')) {
        return;
    }

    // Verificar el modo de bloqueo
    $modo_bloqueo = get_option('mdq_modo_bloqueo', 'todo');

    if ($modo_bloqueo === 'none') {
        return; // No bloquear si está seleccionado "Desbloquear Todo"
    }

    // Obtener mensaje e imagen
    $mensaje = get_option('mdq_mensaje_mantenimiento', 'Estamos en mantenimiento. Volvemos pronto.');
    $imagen = get_option('mdq_imagen_mantenimiento', '');
    $imagen = $imagen ?: plugin_dir_url(__FILE__) . 'assets/images/logo.webp'; // Usa la imagen por defecto si no hay otra

    if ($modo_bloqueo === 'todo') {
        // Bloquear todo el sitio
        wp_die(
            '<div style="text-align: center; padding: 50px;">' .
            ($imagen ? "<img src='" . esc_url($imagen) . "' alt='" . esc_attr__('Mantenimiento', 'mdq-sitio-en-mantenimiento') . "' style='max-width: 100%; margin-bottom: 20px;'>" : '') .
            "<h1>" . esc_html($mensaje) . "</h1>" .
            '</div>',
            esc_html__('Sitio en Mantenimiento', 'mdq-sitio-en-mantenimiento'),
            ['response' => 503]
        );
    }

    if ($modo_bloqueo === 'categorias') {
        $categorias_bloqueadas = get_option('mdq_categorias_bloqueadas', []);
        if (!empty($categorias_bloqueadas) && is_single()) {
            // Obtener las categorías de la entrada actual
            $entrada_categorias = wp_get_post_categories(get_the_ID(), ['fields' => 'ids']);

            // Comprobar si alguna categoría de la entrada está bloqueada
            foreach ($entrada_categorias as $categoria_id) {
                if (in_array($categoria_id, $categorias_bloqueadas)) {
                    wp_die(
                        '<div style="text-align: center; padding: 50px;">' .
                        ($imagen ? "<img src='" . esc_url($imagen) . "' alt='" . esc_attr__('Mantenimiento', 'mdq-sitio-en-mantenimiento') . "' style='max-width: 100%; margin-bottom: 20px;'>" : '') .
                        "<h1>" . esc_html($mensaje) . "</h1>" .
                        '</div>',
                        esc_html__('Sitio en Mantenimiento', 'mdq-sitio-en-mantenimiento'),
                        ['response' => 503]
                    );
                }
            }
        }
    }

    if ($modo_bloqueo === 'paginas') {
        $paginas_bloqueadas = get_option('mdq_paginas_bloqueadas', []);
        if (!empty($paginas_bloqueadas) && is_page()) {
            // Comprobar si la página actual está bloqueada
            if (in_array(get_the_ID(), $paginas_bloqueadas)) {
                wp_die(
                    '<div style="text-align: center; padding: 50px;">' .
                    ($imagen ? "<img src='" . esc_url($imagen) . "' alt='" . esc_attr__('Mantenimiento', 'mdq-sitio-en-mantenimiento') . "' style='max-width: 100%; margin-bottom: 20px;'>" : '') .
                    "<h1>" . esc_html($mensaje) . "</h1>" .
                    '</div>',
                    esc_html__('Sitio en Mantenimiento', 'mdq-sitio-en-mantenimiento'),
                    ['response' => 503]
                );
            }
        }
    }
}


    public function verificar_programacion() {
        $manual = get_option('mdq_modo_manual', false);

        if ($manual) {
            return; // Si el modo manual está activado, no procesar programación.
        }

        $inicio = strtotime(get_option('mdq_inicio_mantenimiento', ''));
        $fin = strtotime(get_option('mdq_fin_mantenimiento', ''));

        if ($inicio && $fin && time() >= $inicio && time() <= $fin) {
            update_option('mdq_mantenimiento_activo', true);
        } else {
            update_option('mdq_mantenimiento_activo', false);
        }
    }

    public function cargar_scripts($hook) {
        if ($hook === 'toplevel_page_mdq-mantenimiento') {
            wp_enqueue_media();
            wp_enqueue_script('mdq-admin-js', plugin_dir_url(__FILE__) . 'assets/js/admin.js', ['jquery'], '1.0', true);
        }
    }
}

new MDQ_Sitio_Mantenimiento();